%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   TP Eben - Berührungslose Messungen für den Bauvertrag
%   wlp_v1.m
    %   Berechnet Bewertetes (WLP)
%   Version 1.0
%   16.03.2023
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function WLP=wlp_v1(x, np_eval, waviness, Lmin, Lmax)
    % Input
    %    x: profile containing np = 2048 samples (sampling interval: 0.1 m)
    %    np_eval: number of samples according to the evaluation length
    %    waviness: slope of the WLP reference Power Spectral Density
    %    Lmin, Lmax: minimum and maximum wavelength for WLP calculation
    % Output
    %    WLP:  WLP contains np_eval data centered at middle of x
    %    delta_WLP: range of WLP based on evaluation length
    %    sigma_WLP: standard deviation of WLP based on evaluation length
    
    
    np = 2048;                      % number of samples
    sample_int = 0.1;               % sampling interval [m]
    nbands = 10;                    % number of octave bands
    lambda_s = np*sample_int;       % maximum wavelength FFT  [m]
    lambda_ratio_L = Lmin/lambda_s;
    lambda_ratio_U = Lmax/lambda_s;

    %%% evaluate low-pass filter (L)
    l = bw_lp(np, lambda_ratio_L, 4);
    
    %%% evaluate high-pass filter (H)
    h = bw_hp(np, lambda_ratio_U, 4);
    
    %%% evaluate weighting function (W)
    w = weight_ft(np, lambda_ratio_U, waviness);
    
%--------------------------------------------------------------------------
    
    %%% E.2 Fourier transformation
    z = fft(x(:));

    %%% E.3 Limiting the wavelength range
    Z = z.*l.*h;

    %%% E.4 Weighting of Fourier transform
    Z_w = w.*Z;

    %%% E.5 Octave-band filtering
    for i = 1:nbands
        Li = bw_lp(np, 2^-i, 4);
        Hi = bw_hp(np, 2^-(i-1.5), 4);
        Z_wo(:, i) = Li.*Hi.*Z_w;
    end

    %%% E.6 Inverse Fourier transform
    z_wo = real(ifft(Z_wo));

    %%% E.7 Weighted Longitudinal Profile computation
    WLP = wlp_comp(z_wo, np_eval);
    
    %%% E.8 Range and Standard Deviation of WLP (computed within another function)    

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% local functions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Butterworth high-pass filter

function H = bw_hp(np, lambda_ratio, n)
% Input
%    np: number of samples for FFT (here: np=2048)
%        must be a power of 2
%    lambda_ratio: ratio of cut-off wavelength and maximum wavelength in
%                  Fourier spectrum
%    n: order of Butterworth high-pass filter (e.g. n=4)
% Output
%    H: evaluated Butterworth high-pass filter (np x 1)

    % compute first half of highpass (array index 1:(np/2+1))
    index = 0:(np/2);
    H = 1 - 1./sqrt(1+(index*lambda_ratio).^(2*n))';
    % mirror function at array index (np/2+1)
    H = [H(1:(np/2+1)); H(np/2:-1:2)];
    
%%% Butterworth low-pass filter

function L = bw_lp(np, lambda_ratio, n)
% Input
%    np: number of samples for FFT (here: np=2048)
%        must be a power of 2
%    lambda_ratio: ratio of cut-off wavelength and maximum wavelength in
%        Fourier spectrum
%    n: order of Butterworth low-pass filter (e.g. n=4)
% Output
%    H: evaluated Butterworth low-pass filter (np x 1)

    % compute first half of lowpass (array index 1:(np/2+1))
    index = 0:(np/2);
    L = 1./sqrt(1+(index*lambda_ratio).^(2*n))';

    % mirror function at array index (np/2+1)
    L = [L(1:(np/2+1)); L(np/2:-1:2)];

%%% Weighting of FT

function W = weight_ft(np, lambda_ratio, w)
% Input
%    np: number of samples for FFT (here: np=2048)
%        must be a power of 2
%    lambda_ratio: ratio of cut-off wavelength and maximum wavelength in
%                  Fourier spectrum
%    w: waviness of the reference spectrum (w = 2.6)
% Output
%    W: weighting vector (double, np x 1)

    % compute first half of weighting function (array index 1:(np/2+1))
    index = 0:(np/2);
    tmp = (w-1)/2;
    num = (index*lambda_ratio).^(w-1);
    denom = 2^tmp - 2^(-tmp);
    W = sqrt(num/denom)';
    % mirror function at array index (np/2+1)
    W = [W(1:(np/2+1)); W(np/2:-1:2)];


%%% Weighted Longitudinal Profile computation

function WLP = wlp_comp(z_wo, np_eval)
% Input
%    z_wo: octave band filtered weighted profiles
%          (np x nbands)
%    np_eval: number of samples according to the evaluation length
% Output
%    WLP: weighted longitudinal profile (np_eval x 1)

    np = size(z_wo, 1);
    shift = round((np-np_eval)/2);

    % symmetrically cut off data
    z_wo_eval = z_wo((shift+1):(shift+np_eval),:);

    % compute standard deviation of sum of octave-filtered profiles
    sigma_tot = std(sum(z_wo_eval,2), 1);
    % compute standard deviation of each octave-filtered profile
    sigma = std(z_wo_eval, 1);

    WLP = sum(z_wo_eval.*repmat(sigma,np_eval,1),2)/sigma_tot;

%%% EOF
