%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%   TP Eben - Berhrungslose Messungen fr den Bauvertrag
    %   Graphische Darstellung des WLP-Schriebes
%   Version 1.1
%   14.05.2024
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear
close all
%--------------------------------------------------------------------------
% Definiere RGB-Codierung der drei farblichen Stufen
%--------------------------------------------------------------------------
Gruenton=[0.5 1 0.6];
Rotton=[1 0.7 0.7];
Gelbton=[0.9 0.9 0];

%--------------------------------------------------------------------------
% Verweise auf Quellverzeichnis Results und gebe dem Nutzer die Mglichkeit
% die Quelldaten zu laden
%--------------------------------------------------------------------------
loc_script=pwd;
loc_data=[loc_script(1:end-29),'Results\'];
[FileName01,PathName01] = uigetfile('*.mat','Importiere WLP-Profile aus 02_WLP_DATA',loc_data,'Multiselect','on');
Zielpfad=[PathName01(1:end-12),'03_WLP_GRAPHS\'];
mkdir(Zielpfad)

%--------------------------------------------------------------------------
% Lade die Daten
% Voraussetzung: Es muss eine Doppelmessung vorliegen
%--------------------------------------------------------------------------
load([PathName01,FileName01{1}])
WLP_Daten1 = WLP_Daten;
clear WLP_Daten
load([PathName01,FileName01{2}])
WLP_Daten2 = WLP_Daten;
clear WLP_Daten

%--------------------------------------------------------------------------
% Lege Grenzwerte fest
%--------------------------------------------------------------------------
GW_VAL.Raster = 2;              % Lnge des Auswerterasters [m]
GW_VAL.SDBL_krit_tol = 4;       % Grenzwert (Manahmenstufe 1) SBL, DBL [mm]
GW_VAL.PGR_krit_tol = 4;        % Grenzwert PGR [mm]
GW_VAL.PGR_mang_tol = 9;        % Grenzwert (Manahmenstufe 2) PGR [mm]
GW_VAL.Schwelle_period = 0.5;   % Grenzwert Pegel [-]

%--------------------------------------------------------------------------
% Definition des 2m-Auswerterasters und des Darstellungsfensters (100m)
%--------------------------------------------------------------------------
RASTERW.nR=GW_VAL.Raster/Profilparameter.dx;        % [-] Anzahl der Punkte pro Raster
RASTERW.NR=floor(Profilparameter.Nwlp/RASTERW.nR);  % [-] Anzahl der Raster insgesamt

DARSTW.LL = 100;                                    % [m] Darstellungslnge
DARSTW.nLL = DARSTW.LL/Profilparameter.dx;          % Anzahl der Punkte im Darstellungsabschnitt
DARSTW.nRR = DARSTW.nLL/RASTERW.nR;                 % Anzahl der Raster im Darstellungsabschnitt
DARSTW.Nxx = floor(Profilparameter.Nwlp/DARSTW.nLL);% Anzahl der Darstellungsfenster

% Bildung des 2m-Auswerterasters
GRWS_Raster = gr_rasterdefinition(GW_VAL.Raster,DARSTW.LL);

%--------------------------------------------------------------------------
% Bestimme die Eingangsgren aus den 2 Messungen fr LP und WLP
%--------------------------------------------------------------------------
% Bildung des Gesamtergebnisses (Minimum aus beiden Messungen)
WLP_Daten.SBL_all = min([WLP_Daten1.SBL_all WLP_Daten2.SBL_all],[],2);
WLP_Daten.DBL_all = min([WLP_Daten1.DBL_all WLP_Daten2.DBL_all],[],2);
WLP_Daten.SDBL_all= max([WLP_Daten.SBL_all WLP_Daten.DBL_all],[],2);
WLP_Daten.loc_all = WLP_Daten2.loc_all;
WLP_Daten.h_all   = max([WLP_Daten1.h_all WLP_Daten2.h_all],[],2);
station_FE0 = union(WLP_Daten1.loc_Fug_Elim, WLP_Daten2.loc_Fug_Elim);
station_FE = station_FE0(station_FE0 <= WLP_Daten.loc_all(end));
%--------------------------------------------------------------------------
% Bilde die resultierenden Ergebnisse aus den 2 Messungen
%--------------------------------------------------------------------------
% Ermittle Mittelwert der 2 PGR-Schriebe und stelle Mittelwert als
% Rechteckfunktion dar. Berechne deren Toleranzberschreitungen
PGRin = [WLP_Daten.loc_all WLP_Daten1.Plano_all WLP_Daten2.Plano_all];
[p_PGR_krit_rec,PGR_rec] = Calculate_p_PGR(PGRin,GW_VAL.Raster,GW_VAL.PGR_krit_tol);

% Ermittle Maximum von SDBL im 2m-Raster und stelle SDBL als
% Rechteckfunktion dar. Berechne deren Toleranzberschreitungen
SDBLin = [WLP_Daten.loc_all WLP_Daten.SDBL_all];
[p_SDBL_krit_rec,SDBL_rec] = Calculate_p_SDBL(SDBLin,GW_VAL.Raster,GW_VAL.SDBL_krit_tol);

% Bilde Mittelwert aus PEGEL1 und PEGEL2
PEGEL = mean([WLP_Daten1.PEGEL WLP_Daten2.PEGEL],2);

% Bestimme die Wellenlnge ber Gleitenden Median ber 50m
WL0 = [WLP_Daten1.PER WLP_Daten2.PER]';
WL1 = reshape(WL0,1,[]);
WL2 = movmedian(WL1,1000);  % Basis 1000 Werte
WL = WL2(2:2:end)';         % d.h. Basis = 50m Mittelungslnge

%--------------------------------------------------------------------------
% Ermittle die Grenzwertberschreitungen der Ergebnisse
%--------------------------------------------------------------------------
% Bilde Mangelgrenze fr SDBL gem den bilinearen Interpolationsfunktionen
SDBL_mang_tol=per_abschnitte(WLP_Daten.loc_all,PEGEL,GW_VAL.Raster); % [mm]
% Ermittle die Grenzwertberschreitungen (Mangel) fr SDBL
SDBLin_rec = [WLP_Daten.loc_all SDBL_rec];
p_SDBL_mang_rec=Calculate_p_Mangel(SDBLin_rec,GW_VAL.Raster,SDBL_mang_tol);

% Ermittle die Grenzwertberschreitungen (Mangel) fr PGR
PGRin_rec = [WLP_Daten.loc_all PGR_rec];
PGR_mang_tol = GW_VAL.PGR_mang_tol*ones(length(PGR_rec),1);
p_PGR_mang_rec=Calculate_p_Mangel(PGRin_rec,GW_VAL.Raster,PGR_mang_tol);

% Ermittle die Grenzwertberschreitungen fr PEGEL
PERin = [WLP_Daten.loc_all PEGEL];
p_PER = Calculate_p_PER(PERin,GW_VAL.Raster,GW_VAL.Schwelle_period);
% Ermittle die dazugehrige Wellenlnge
p_WL = [];  ind = p_PER(:,1)/Profilparameter.dx + 1;
p_WL(:,1) = p_PER(:,1);
p_WL(:,2) = round(10*WL(ind))/10;

%--------------------------------------------------------------------------
% Markiere die Grenzwertberschreitungen mit gelb oder rot
%--------------------------------------------------------------------------
% WLP gelb und rot
[SDBgl_rec,INDgl_rec]=gr_farbmarkierer([WLP_Daten.loc_all,SDBL_rec],p_SDBL_krit_rec,RASTERW);
[SDBr_rec,INDr_rec]=gr_farbmarkierer([WLP_Daten.loc_all,SDBL_rec],p_SDBL_mang_rec,RASTERW);
% PGR gelb und rot
[PGRgl_rec,INDpgr_gl_rec]=gr_farbmarkierer([WLP_Daten.loc_all,PGR_rec],p_PGR_krit_rec,RASTERW);
[PGRr_rec,INDpgr_r_rec]=gr_farbmarkierer([WLP_Daten.loc_all,PGR_rec],p_PGR_mang_rec,RASTERW);
% Hhenlngsprofil gelb oder rot (orientiert sich an Farbe fr SDBL)
h_allgl = WLP_Daten.h_all(INDgl_rec);
h_allr  = WLP_Daten.h_all(INDr_rec);

%--------------------------------------------------------------------------
% Stelle die Ergebnisse graphisch dar
%--------------------------------------------------------------------------

% Darstellungsloop
for l = 1:DARSTW.Nxx
    
    %----------------------------------------------------------------------
    % Extrahiere die Daten fr die jeweiligen 100m-Darstellungsfenster
    %----------------------------------------------------------------------
    loc = WLP_Daten.loc_all((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    hp1 = WLP_Daten1.h_all((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    hp2 = WLP_Daten2.h_all((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    hp  = WLP_Daten.h_all((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    WLPx1 = WLP_Daten1.WLP_all((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    WLPx2 = WLP_Daten2.WLP_all((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    PEG  = PEGEL((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    SDBL_MANG_SCHRIEB = SDBL_mang_tol((l-1)*DARSTW.nLL+1:l*DARSTW.nLL,1);
    SDBLrec = SDBL_rec((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    PGRrec = PGR_rec((l-1)*DARSTW.nLL+1:l*DARSTW.nLL);
    
    %----------------------------------------------------------------------
    % Positioniere die Textboxen im Bereich der Grenzwertberschreitungen
    %----------------------------------------------------------------------
    % Wellenlnge
    POS_WL = gr_auszuege(loc,p_WL);
    % SDBL
    POS_1l_rec = gr_auszuege(loc,p_SDBL_krit_rec);
    POS_1ly_rec = min([POS_1l_rec(:,2)+GW_VAL.SDBL_krit_tol+2,17*ones(size(POS_1l_rec,1),1)],[],2);
    % PGR
    POS_PGR_rec  = gr_auszuege(loc,p_PGR_krit_rec);
    POS_PGRy_rec = min([POS_PGR_rec(:,2)+GW_VAL.PGR_krit_tol+2,17*ones(size(POS_PGR_rec,1),1)],[],2);
    
    %----------------------------------------------------------------------
    % Bestimme die Farbmarkierung (gelb/rot) im 100m-Darstellungsfenster
    %----------------------------------------------------------------------
    % HLP
    AR_HLPgl=gr_auszuege(loc,[SDBgl_rec(:,1),h_allgl]); % gelb
    AR_HLPr=gr_auszuege(loc,[SDBr_rec(:,1),h_allr]);    % rot
    % SDBL
    AR_SDBLgl_rec=gr_auszuege(loc,SDBgl_rec);           % gelb
    AR_SDBLr_rec=gr_auszuege(loc,SDBr_rec);             % rot
    % PGR
    AR_PGRgl_rec = gr_auszuege(loc,PGRgl_rec);          % gelb
    AR_PGRr_rec = gr_auszuege(loc,PGRr_rec);            % rot
    
    %----------------------------------------------------------------------
    % Bestimme den Dateinamen fr die Speicherung der 100m-Darstellung
    %----------------------------------------------------------------------
    str = num2str((l+0.1)/1000);
    name1 = [FileName01{2}(1:7),'-',str(3:end-1)];
    
    %----------------------------------------------------------------------
    % Setze die x-/y-Achsen fr die Darstellung von LP und WLP fest
    %----------------------------------------------------------------------
    x = 0.1:0.1:DARSTW.LL;
    % Setze Grenzen fr den Plot des HLP
    hpmin = -40;
    hpmax = 60;
    if max(hp)>hpmax-20
        hpmax = ceil(max(hp))+40;
    end
    if min(hp)<hpmin
        hpmin = floor(min(hp));
    end
    % Setze Grenzen fr den Plot des WLP
    y_lim = 18;
    
    %----------------------------------------------------------------------
    % Plotte das 100m-Darstellungsfenster
    %----------------------------------------------------------------------
    %% HLP/WLP-Zeile
    fig = figure;
    subplot(4,1,1)
    
    % Setze Graphen und Raster
    % Definiere das hinterlegte Grau-Wei Raster
    hl = area(GRWS_Raster(1,:),-GRWS_Raster(2,:));
    set(hl,'FaceColor',[0.6 0.6 0.6],'EdgeColor',[0.6 0.6 0.6],'LineWidth',1)
    hold on
    
    % Definiere grne Bereiche (Standardwert: alles)
    HPGRUEN = bar(x,hp,'FaceColor',Gruenton,'EdgeColor','none');
    HPGRUEN.BaseValue = hpmin;
    hold on
    % Definiere gelbe Bereiche (gem Vorberechnung, s.o.)
    HPGELB = bar(AR_HLPgl(:,1),AR_HLPgl(:,2),'FaceColor',Gelbton,'EdgeColor','none');
    HPGELB.BaseValue = hpmin;
    hold on
    % Definiere rot Bereiche (gem Vorberechnung, s.o.)
    HPROT = bar(AR_HLPr(:,1),AR_HLPr(:,2),'FaceColor',Rotton,'EdgeColor','none');
    HPROT.BaseValue = hpmin;
    hold on
    
    % Zeichne HLP-Ausschnitt, setze Gitter
    HP1LGD = plot(x,hp1,'k','LineWidth',0.5);
    hold on
    HP2LGD = plot(x,hp2,'k--','LineWidth',0.5);
    hold on
    
    % Definiere die Referenzebene  (Profilhhe = 0)
    plot([0 DARSTW.LL],[0 0],'k')
    hold on
    
    % Zeichne Einhllenden (Andeutung des DBL-Grenzwertes)
    plot([0 DARSTW.LL],GW_VAL.SDBL_krit_tol*2*[1 1],'k:','lineWidth',0.5);
    hold on
    plot([0 DARSTW.LL],GW_VAL.SDBL_krit_tol*2*[-1 -1],'k:','lineWidth',0.5);
    
    
    %Achsenparameter (X-Achse und Y-Achse (links))
    axis([0 DARSTW.LL hpmin hpmax])
    ylabel('Hhe [mm]','FontSize',8)
    set(gca,'XColor','k','YColor','k');
    set(gca,'FontSize',8)
    
    % Zeichne WLP-Ausschnitt
    WP1LGD = plot(x,WLPx1,'r','LineWidth',0.5);
    hold on
    WP2LGD = plot(x,WLPx2,'Color',[1 0.4 0.4],'LineWidth',0.5);
    hold on
    yyaxis right
    % Achsenparameter (Y-Achse (rechts))
    ylim([hpmin hpmax])
    set(gca,'XColor','k','YColor','r');
    set(gca,'FontSize',8)
    ylabel('WLP [mm]','FontSize',8)
    
    lgd = legend([HP1LGD,HP2LGD,WP1LGD,WP2LGD,HPGRUEN,HPGELB,HPROT],'LP1; ','LP2; ','WLP1; ','WLP2; ','i.O.;  ','Manahmenstufe 1;  ','Manahmenstufe 2','Location','north','Orientation','horizontal');
    lgd.FontSize = 5;
    
    
    %% SBL/DBL-Zeile
    subplot(4,1,2)
    
    % Rasterplot
    h = area(GRWS_Raster(1,:),GRWS_Raster(2,:));
    set(h,'FaceColor',[0.6 0.6 0.6],'EdgeColor',[0.6 0.6 0.6],'LineWidth',1)
    hold on
    
    % Farbplot
    SBLGRUEN=bar(x,SDBLrec,'FaceColor',Gruenton,'EdgeColor','none');
    hold on;
    SBLGELB=bar(AR_SDBLgl_rec(:,1),AR_SDBLgl_rec(:,2),'FaceColor',Gelbton,'EdgeColor','none');
    hold on;
    SBLROT=bar(AR_SDBLr_rec(:,1),AR_SDBLr_rec(:,2),'FaceColor',Rotton,'EdgeColor','none');
    hold on;
    
    % SBL-Graphenplot und Achsbeschriftung (x-Achse sowie y-links)
    SDBLKRV=plot(x,SDBLrec,'b','LineWidth',1);
    hold on
    axis([0 DARSTW.LL 0 y_lim])
    ylabel('WLP\_MAX 2m [mm]','FontSize',8)
    set(gca, 'YTick',[0 GW_VAL.SDBL_krit_tol y_lim/2 y_lim])
    set(gca,'XColor','k','YColor','k');
    set(gca,'FontSize',8)
    
    % Einziehen der Grenzwertlinien
    % Abnahmewert
    ABNAHML=plot([0 DARSTW.LL],[GW_VAL.SDBL_krit_tol GW_VAL.SDBL_krit_tol],'k-','lineWidth',1);
    hold on
    % Mngelbeseitigungswert
    plot(x,SDBL_MANG_SCHRIEB,'k-','lineWidth',1);
    hold on
    
    % Zahlenmige Darstellung der SBL/DBL-Werte bei berschreitung
    for j1 = 1:length(POS_1l_rec(:,1))
        text(POS_1l_rec(j1,1)-GW_VAL.Raster/4,POS_1ly_rec(j1),...
            num2str(POS_1l_rec(j1,2)+GW_VAL.SDBL_krit_tol),'FontSize',6,'Color','b')
    end
    
    % Gitter und Legende
    grid
    set(gca,'GridLineStyle','-')
    set(gca,'YGrid','on')
    set(gca, 'YTick',[0 GW_VAL.SDBL_krit_tol y_lim/2 y_lim])
    lgd = legend([SDBLKRV,ABNAHML,SBLGRUEN,SBLGELB,SBLROT],'max(SBL;DBL/4);  ','GW; ','i.O.;  ','Manahmenstufe 1;  ','Manahmenstufe 2','Location','north','Orientation','horizontal');
    lgd.FontSize = 5;
    
    %% Pegel-Zeile
    subplot(4,1,3)
    
    % Rasterplot
    h = area(GRWS_Raster(1,:),GRWS_Raster(2,:));
    set(h,'FaceColor',[0.6 0.6 0.6],'EdgeColor',[0.6 0.6 0.6],'LineWidth',1)
    hold on
    
    % Farbplot
    bar(x,PEG,'FaceColor',[0.6 0.6 0.6],'EdgeColor','none');
    hold on
    
    % Periodizittenlinie und Achsbeschriftung (x-Achse sowie y-links)
    plot(x,PEG,'k','LineWidth',0.5);
    hold on
    axis([0 DARSTW.LL 0 1.25])
    ylabel('Pegel [-]','FontSize',8)
    
    % Einziehen der Grenzwertlinien
    plot([0 DARSTW.LL],[GW_VAL.Schwelle_period GW_VAL.Schwelle_period],'k','lineWidth',1);
    hold on
    plot([0 DARSTW.LL],[1 1],'k:','lineWidth',0.5);
    
    % Zahlenmige Darstellung der Wellenlnge bei Pegel-berschreitung
    text(50,1.1,['\color{black} Pegelwerte > 0.5 mit Ausweisung der zugrunde liegenden dominanten Periodenlnge (oder deren','\color{red} Vielfache','\color{black}) in Meter'],'FontSize',8,'HorizontalAlignment','center')
    for j1 = 1:2:length(POS_WL(:,1))
        text(POS_WL(j1,1)-GW_VAL.Raster/2,0.1,num2str(POS_WL(j1,2)),'FontSize',6,'Color','k')
    end
    for j2 = 2:2:length(POS_WL(:,1))
        text(POS_WL(j2,1)-GW_VAL.Raster/2,0.2,num2str(POS_WL(j2,2)),'FontSize',6,'Color','k')
    end
    % Gitter
    grid
    set(gca,'GridLineStyle','-')
    set(gca,'YGrid','on')
    set(gca, 'YTick',[0 0.5 1])
    set(gca,'XColor','k','YColor','k');
    set(gca,'FontSize',8)
    
    %% PGR-Zeile
    subplot(4,1,4)
    
    % Rasterplot
    h = area(GRWS_Raster(1,:),GRWS_Raster(2,:));
    set(h,'FaceColor',[0.6 0.6 0.6],'EdgeColor',[0.6 0.6 0.6],'LineWidth',1)
    hold on
    
    % Farbplot
    PGRGRUEN=bar(x,PGRrec,'FaceColor',Gruenton,'EdgeColor','none');
    hold on
    PGRGELB=bar(AR_PGRgl_rec(:,1),AR_PGRgl_rec(:,2),'FaceColor',Gelbton,'EdgeColor','none');
    hold on
    PGRROT=bar(AR_PGRr_rec(:,1),AR_PGRr_rec(:,2),'FaceColor',Rotton,'EdgeColor','none');
    hold on
    
    % Planographenschrieb und Achsbeschriftung (x-Achse sowie y-links)
    PGRKRV=plot(x,PGRrec,'b','LineWidth',1);
    hold on
    axis([0 DARSTW.LL 0 y_lim])
    ylabel('PGR\_MAX 2m [mm]','FontSize',8)
    xlabel('Weg [m]','FontSize',8)
    set(gca, 'YTick',[0 GW_VAL.PGR_krit_tol y_lim/2 y_lim])
    set(gca,'XColor','k','YColor','k');
    set(gca,'FontSize',8)
    
    % Einziehen der Grenzwertlinien
    % Abnahmewert
    ABNAHML=plot([0 DARSTW.LL],[GW_VAL.PGR_krit_tol GW_VAL.PGR_krit_tol],'k','lineWidth',1);
    hold on
    % Mngelbeseitigungswert
    plot([0 DARSTW.LL],[9 9],'k','lineWidth',1);
    hold on
    
    % Zahlenmige Darstellung der PGR-Werte bei berschreitung
    for j1 = 1:length(POS_PGR_rec(:,1))
        text(POS_PGR_rec(j1,1)-GW_VAL.Raster/4,POS_PGRy_rec(j1),num2str(POS_PGR_rec(j1,2)+GW_VAL.PGR_krit_tol),'FontSize',6,'Color','b')
    end
    % Gitter und Legende
    grid
    set(gca,'GridLineStyle','-')
    set(gca,'YGrid','on')
    set(gca, 'YTick',[0 GW_VAL.PGR_krit_tol y_lim/2 y_lim])
    lgd = legend([PGRKRV,ABNAHML,PGRGRUEN,PGRGELB,PGRROT],'avg(PGR1;PGR2);  ','GW; ','i.O.;  ','Manahmenstufe 1;  ','Manahmenstufe 2','Location','north','Orientation','horizontal');
    lgd.FontSize = 5;
    
    saveas (fig,[Zielpfad,'GR_REC_',name1,'_DM'],'pdf')
    close all
    
end

disp(['Abgeschlossen: ',FileName01{2}(1:7)])

%--------------------------------------------------------------------------
% Erstelle Ergebnistabelle mit berschreitungen p und p^2
%--------------------------------------------------------------------------
% Bilde den Tabellennamen
index = find(name1=='-');
Name = [Zielpfad,'Ergebnis_',name1(1:index-1),'.xlsx'];

% Bilde die Tabelle fr das WLP
station_wlp = p_SDBL_krit_rec(:,1);
p_wlp  = p_SDBL_krit_rec(:,2);
p2_wlp = p_SDBL_krit_rec(:,2).^2;

% Bilde die Tabelle fr PGR
station_pgr = p_PGR_krit_rec(:,1);
p_pgr  = p_PGR_krit_rec(:,2);
p2_pgr = p_PGR_krit_rec(:,2).^2;

% Bilde die Bezeichnung fr die Tabellen-Spalten
varNames = {'Station [m]','p [mm]','p^2 [mm^2]'};
% Speichere die Tabelle fr das WLP ab
T_wlp = table(station_wlp,p_wlp,p2_wlp,'VariableNames',varNames);
writetable(T_wlp,Name,'Sheet','WLP')

% Speichere die Tabelle fr PGR ab
T_pgr = table(station_pgr,p_pgr,p2_pgr,'VariableNames',varNames);
writetable(T_pgr,Name,'Sheet','PGR')

% Speichere die Tabelle fr die vorgenommenen Fugeneliminierungen ab
T_FEL = table(station_FE,'VariableNames',varNames(1));
writetable(T_FEL,Name,'Sheet','Fugeneliminierung')

%%% EOF
